/// Provides a standard implementation of `QueryId`.
/// Apps should not need to concern themselves with this macro.
///
/// This macro should be called with the name of your type, along with all of
/// it's type parameters.
/// If the SQL generated by your type not uniquely identifiable by the type
/// itself, you should put `noop:` in front.
///
/// For example, given the type `And<Left, Right>`, invoking
/// `impl_query_id!(And<Left, Right>)` will generate:
///
/// ```rust,ignore
/// impl<Left, Right> QueryId for And<Left, Right>
/// where
///     Left: QueryId,
///     Right: QueryId,
/// {
///     type QueryId = And<Left::QueryId, Right::QueryId>;
///
///     const HAS_STATIC_QUERY_ID: bool = Left::HAS_STATIC_QUERY_ID && Right::HAS_STATIC_QUERY_ID;
/// }
/// ```
///
/// Invoking `impl_query_id!(noop: And<Left, Right>)` will generate:
///
/// ```rust,ignore
/// impl<Left, Right> QueryId for And<Left, Right> {
///     type QueryId = ();
///
///     const HAS_STATIC_QUERY_ID: bool = false;
/// }
/// ```
#[macro_export]
#[cfg(feature = "with-deprecated")]
#[deprecated(since = "1.1.0", note = "Use `#[derive(QueryId)]` instead")]
macro_rules! impl_query_id {
    ($name: ident) => {
        impl $crate::query_builder::QueryId for $name {
            type QueryId = Self;

            const HAS_STATIC_QUERY_ID: bool = true;
        }
    };

    ($name: ident<$($ty_param: ident),+>) => {
        #[allow(non_camel_case_types)]
        impl<$($ty_param),*> $crate::query_builder::QueryId for $name<$($ty_param),*> where
            $($ty_param: $crate::query_builder::QueryId),*
        {
            type QueryId = $name<$($ty_param::QueryId),*>;

            const HAS_STATIC_QUERY_ID: bool = $($ty_param::HAS_STATIC_QUERY_ID &&)* true;
        }
    };

    (noop: $name: ident) => {
        impl $crate::query_builder::QueryId for $name {
            type QueryId = ();

            const HAS_STATIC_QUERY_ID: bool = false;
        }
    };

    (noop: $name: ident<$($ty_param: ident),+>) => {
        #[allow(non_camel_case_types)]
        impl<$($ty_param),*> $crate::query_builder::QueryId for $name<$($ty_param),*> {
            type QueryId = ();

            const HAS_STATIC_QUERY_ID: bool = false;
        }
    }
}
